/*
  ==============================================================================

    SonicCrypt Flux Gate
    Copyright (C) 2025 Sebastian Sünkler

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.

  ==============================================================================
*/

#pragma once
#include <JuceHeader.h>
#include <vector>

class SonicCryptFluxGateAudioProcessor : public juce::AudioProcessor, public juce::ValueTree::Listener
{
public:
    SonicCryptFluxGateAudioProcessor();
    ~SonicCryptFluxGateAudioProcessor() override;

    void prepareToPlay(double sampleRate, int samplesPerBlock) override;
    void releaseResources() override;
    bool isBusesLayoutSupported(const BusesLayout& layouts) const override;
    void processBlock(juce::AudioBuffer<float>&, juce::MidiBuffer&) override;

    juce::AudioProcessorEditor* createEditor() override;
    bool hasEditor() const override;

    const juce::String getName() const override;
    bool acceptsMidi() const override;
    bool producesMidi() const override;
    bool isMidiEffect() const override;
    double getTailLengthSeconds() const override;

    int getNumPrograms() override;
    int getCurrentProgram() override;
    void setCurrentProgram(int index) override;
    const juce::String getProgramName(int index) override;
    void changeProgramName(int index, const juce::String& newName) override;

    void getStateInformation(juce::MemoryBlock& destData) override;
    void setStateInformation(const void* data, int sizeInBytes) override;

    juce::AudioProcessorValueTreeState& getAPVTS() { return apvts; }

    juce::File getPresetFolder() const;
    void savePreset(const juce::String& name);
    void loadPresetFile(const juce::File& file);

    // GUI Data
    int lastUIWidth = 1000;
    int lastUIHeight = 600;

    std::vector<float> currentPattern;
    int currentStepIndex = -1;

    void togglePatternStep(int stepIndex);
    void setPatternStepLevel(int stepIndex, float level);
    void triggerFluxRandomization();

private:
    juce::AudioProcessorValueTreeState apvts;
    juce::AudioProcessorValueTreeState::ParameterLayout createParameterLayout();
    void valueTreePropertyChanged(juce::ValueTree&, const juce::Identifier&) override;

    // NEU: Hilfsfunktion für Pattern-Updates mit Gedächtnis
    void updatePattern(int steps, int pulses, int rot);

    // Audio Engine
    juce::LinearSmoothedValue<float> gateSmootherL;
    juce::LinearSmoothedValue<float> gateSmootherR;
    float currentSmoothMs = -1.0f;

    // Trigger Logic
    double beatPhasor = 0.0;
    bool isTriggered = false;
    int releaseTimer = 0;
    int activeMidiNotes = 0;

    // Safety Vars
    bool isSilent = true;
    int silenceCounter = 0;

    // Stutter Engine
    juce::AudioBuffer<float> stutterBuffer;
    int stutterWritePos = 0;
    bool isStuttering = false;
    int stutterReadOffset = 0;
    int stutterLoopLength = 0;
    int stutterCooldown = 0;
    float stutterFade = 0.0f;

    // NEU: Das Gedächtnis für die Lautstärke
    std::vector<float> velocityMap;

    // Cache
    int lastSteps = -1;
    int lastPulses = -1;
    int lastRot = -999;

    JUCE_DECLARE_NON_COPYABLE_WITH_LEAK_DETECTOR(SonicCryptFluxGateAudioProcessor)
};